function [loss_dist_inter, num_available] = port_loss_dist_interpolate(dates, T, portfolio, index_members, y0, ajd_common_factor, N, freq, LGD_mean)
% --------------------------------------------------------------------------------------------------
% Same as 'port_loss_dist', but use geometric interpolation to speed things up.
% --------------------------------------------------------------------------------------------------
% dates             ... dates for which to compute future portfolio loss distribution
% T                 ... horizons for which to compute future portfolio loss distribution
% portfolio         ... portfolio of credit default swaps
% index_members     ... indicator matrix, i-th row representing members of index for i-th date
% y0                ... initial intensity of common factor
% ajd_common_factor ... AJD dynamics of the common factor
% N                 ... number of points for numerical integration of transform
% freq              ... spacing of interpolation points, e.g 4=use every 4th horizons
% LGD_mean          ... mean LGD
% --------------------------------------------------------------------------------------------------
% sample call: port_loss_dist_interpolate(cdx_test.dates{1}, [1 5; 2 4], cdx_test.portfolio, cdx_test.index_members, [0.01; 0.01], ajd_dynamics, 1000, 2)
%              port_loss_dist_interpolate(cdx_test.dates{1}, [0:0.25:5; 0:0.25:5], cdx_test.portfolio, cdx_test.index_members, [0.01; 0.01], ajd_dynamics, 1000, 4)
% --------------------------------------------------------------------------------------------------

% Select subset of horizons with appropriate spacing
num_horizons = size(T, 2);
num_dates = length(dates);
used_horizons = num_horizons:(-freq):1;
used_horizons = used_horizons(length(used_horizons):(-1):1);
% Always include first horizon
if (used_horizons(1) ~= 1)
    used_horizons = [1 used_horizons];
end
not_used_horizons = find(~logical(is_member_sorted_c(1:num_horizons, used_horizons)));

% If first horizon == 0, then don't need to caculate
if (sum(T(:,1)) == 0)
    % Calculate portfolio loss distribution on spare set of horizons
    [loss_dist, num_available] = port_loss_dist3(dates, T(:,used_horizons(2:end)), portfolio, index_members, y0, ajd_common_factor, N);
    if (min(min(min(loss_dist))) < -1e-6)
        error('Loss distribution has negative values');
    else
        loss_dist = max(0, loss_dist);
    end
    tmp = zeros(size(loss_dist) + [0 1 0]);
    tmp(:,1,1) = ones(num_dates, 1);
    tmp(:,2:end,:) = loss_dist;
    loss_dist = tmp;
else
    [loss_dist, num_available] = port_loss_dist2(dates, T(:,used_horizons), portfolio, index_members, y0, ajd_common_factor, N, LGD_mean);
end

% % Calculate mean and standard deviation of loss distributions
% means = zeros(num_dates, length(used_horizons));
% std_devs = zeros(num_dates, length(used_horizons));
% for j=1:num_dates
%     for k=1:length(used_horizons)
%         max_m = size(loss_dist,3)-1;
%         means(j,k) = sum(squeeze(loss_dist(j,k,:))' .* (0:max_m));
%         second_moment = sum(squeeze(loss_dist(j,k,:))' .* (0:max_m).^2);
%         std_devs(j,k) = sqrt(second_moment - means(j,k)^2);
%     end
% end
% plot(T(1,used_horizons), means(1,:));
% plot(T(1,used_horizons), std_devs(1,:));

% % Use cubic spline interpolation to estimate mean and standard deviation of loss distribution at
% % non-grid points
% max_horizon = max(max(used_horizons));
% means_all = zeros(num_dates, max_horizon);
% std_devs_all = zeros(num_dates, max_horizon);
% for j=1:num_dates
%     % Calculate mean and std_dev
%     means_all(j,:) = max(0, interp1(T(j, used_horizons), means(j,:), T(j,:), 'spline'));
%     std_devs_all(j,:) = max(0, interp1(T(j, used_horizons), std_devs(j,:), T(j,:), 'spline'));
%     
%     % For monotonicity
%     means_all(j,:) = cum_max_c(means_all(j,:));
%     std_devs_all(j,:) = cum_max_c(std_devs_all(j,:));
% end

% % Use geometric interpolation to obtain estimate of portfolio loss distribution for all horizons
% num_dates = size(loss_dist, 1);
% max_firms = max(num_available);
% loss_dist_inter = zeros(num_dates, size(T, 2), max_firms+1);
% loss_dist_inter(:,used_horizons,:) = loss_dist;
% for i=1:length(not_used_horizons)
%     % Determine closest available dates to be used for interpolation
%     j = not_used_horizons(i);
%     last_available = find(used_horizons <= j, 1, 'last');
%     last_available = used_horizons(last_available);
%     next_available = find(used_horizons >= j, 1, 'first');
%     next_available = used_horizons(next_available);
%     
%     % Determine weighting factor for interpolation
%     a = T(:,last_available);
%     b = T(:,j);
%     c = T(:,next_available);
%     omega = (c-b) ./ (c-a);
%     omega = omega(:,ones(1, max_firms+1));
%     
%     % Do geometric interpolation
%     density_a = reshape(loss_dist_inter(:,last_available,:), [num_dates max_firms+1]);
%     density_c = reshape(loss_dist_inter(:,next_available,:), [num_dates max_firms+1]);
%     density_b = density_a.^omega .* density_c.^(1-omega);
%     % If a) non-overlapping support or b) last_available==1, then use linear interpolation instead
%     if ((last_available==1) | (min(sum(density_b,2)) == 0))
%         density_b = omega.*squeeze(density_a) + (1-omega).*squeeze(density_c);
%     else
%         tmp = sum(density_b, 2);
%         density_b = density_b ./ tmp(:,ones(1,max_firms+1));
%     end
%     if (0)
%         density_b = omega.*squeeze(density_a) + (1-omega).*squeeze(density_c);
%     end
%     if (0)  % Independent Poisson convolution
%         % Estimate mu_b
%         mu_a = sum(density_a .* (0:max_firms));
%         std_a = sqrt(sum(density_a .* (0:max_firms).^2) - mu_a.^2);
%         mu_c = sum(density_c .* (0:max_firms));
%         std_c = sqrt(sum(density_c .* (0:max_firms).^2) - mu_c.^2);
%         %mu_b = omega(1) * mu_a + (1-omega(1)) * mu_c;
%         mu_b = means_all(:,j);
%         
%         % Poisson convolution
%         P_dist = poisspdf(0:max_firms, mu_b - mu_a);
%         density_b = conv(density_a, P_dist);
%         density_b = density_b(1:(max_firms+1));
%     end
%     if (0)  % Overdispersed Poisson convolution
%         % Calculate lambda_1 and lambda_2 for Poisson micture
%         mu_a = sum(density_a .* (0:max_firms));
%         std_a = sqrt(sum(density_a .* (0:max_firms).^2) - mu_a.^2);
%         mu_b = means_all(:,j);
%         std_b = std_devs_all(:,j);
%         p = 1/3;    % mixture probability
%         lambda_2 = min((mu_b - mu_a)/(1-p), max(0, (p*(mu_b - mu_a) - (std_b.^2 - std_a.^2)) / (-1+3*p-2*p^2)));
%         lambda_1 = max(1e-10, ((mu_b - mu_a) - (1-p)*lambda_2) / p);
%         
%         % Poisson convolution
%         P_dist = p * poisspdf(0:max_firms, lambda_1) + (1-p) * poisspdf(0:max_firms, lambda_2);
%         density_b = conv(density_a, P_dist);
%         density_b = density_b(1:(max_firms+1));        
%     end
%     loss_dist_inter(:,j,:) = density_b;
% end

% Use spline interpolation to obtain estimate of portfolio loss distribution for all horizons
num_dates = size(loss_dist, 1);
max_firms = max(num_available);
loss_dist_inter = zeros(num_dates, size(T, 2), max_firms+1);
loss_dist_inter(:,used_horizons,:) = loss_dist;
for j=1:num_dates;
    loss_interp = min(1, max(0, interp1(T(j, used_horizons)', squeeze(loss_dist_inter(j, used_horizons, :)), T(j,:)', 'spline')));
    loss_dist_inter(j, 1:size(loss_interp, 1), 1:size(loss_interp, 2)) = loss_interp;
end
        
% Normalize density
for j=1:num_dates;
    for t=1:max(used_horizons)
        loss_dist_inter(j, t, :) = loss_dist_inter(j, t, :) / sum(loss_dist_inter(j, t, :));
    end
end

% Do couple of checks that interpolation makes sense
% disp(squeeze(loss_dist_inter(1,:,:)));
